import os
import json

# Base paths
BASE_PATH = "."
VARIATIONS_PATH = os.path.join(BASE_PATH, "assets/pokemonleague/bedrock/npcs/variations")
NPCS_PATH = os.path.join(BASE_PATH, "data/pokemonleague/npcs")

# NPC definitions
NPCS = [
    # Gym Leaders
    {"id": "gym_leader_rock", "name": "Pierrette", "skill": 4},
    {"id": "gym_leader_water", "name": "Ondine", "skill": 4},
    {"id": "gym_leader_electric", "name": "Major Bob", "skill": 4},
    {"id": "gym_leader_grass", "name": "Erika", "skill": 5},
    {"id": "gym_leader_psychic", "name": "Morgane", "skill": 5},

    # Rock Gym Trainers
    {"id": "gym_trainer_rock_1", "name": "Randonneur Luc", "skill": 2},
    {"id": "gym_trainer_rock_2", "name": "Montagnard Pierre", "skill": 2},
    {"id": "gym_trainer_rock_3", "name": "Speleologue Marc", "skill": 3},

    # Water Gym Trainers
    {"id": "gym_trainer_water_1", "name": "Nageur Thomas", "skill": 2},
    {"id": "gym_trainer_water_2", "name": "Nageuse Marie", "skill": 3},
    {"id": "gym_trainer_water_3", "name": "Pecheur Jacques", "skill": 3},

    # Electric Gym Trainers
    {"id": "gym_trainer_electric_1", "name": "Ingenieur Paul", "skill": 3},
    {"id": "gym_trainer_electric_2", "name": "Rocker Julien", "skill": 3},
    {"id": "gym_trainer_electric_3", "name": "Scientifique Emma", "skill": 3},

    # Grass Gym Trainers
    {"id": "gym_trainer_grass_1", "name": "Botaniste Claire", "skill": 3},
    {"id": "gym_trainer_grass_2", "name": "Jardiniere Sophie", "skill": 3},
    {"id": "gym_trainer_grass_3", "name": "Aromatherapeute Lea", "skill": 4},

    # Psychic Gym Trainers
    {"id": "gym_trainer_psychic_1", "name": "Medium Antoine", "skill": 3},
    {"id": "gym_trainer_psychic_2", "name": "Voyante Luna", "skill": 4},
    {"id": "gym_trainer_psychic_3", "name": "Telepathe Maxime", "skill": 4},

    # League Champions
    {"id": "league_champion_1", "name": "Pyroclaste", "skill": 3},
    {"id": "league_champion_2", "name": "Sylvestre", "skill": 3},
    {"id": "league_champion_3", "name": "Geolithe", "skill": 4},
    {"id": "league_champion_4", "name": "Entomologue", "skill": 4},
    {"id": "league_champion_5", "name": "Aquatique", "skill": 4},
    {"id": "league_champion_6", "name": "Obscur", "skill": 5},
    {"id": "league_champion_7", "name": "Maitresse Dragon", "skill": 5},
]

def create_variation(npc_id):
    """Create variation JSON file"""
    variation = {
        "name": f"pokemonleague:{npc_id}",
        "order": 0,
        "variations": [
            {
                "aspects": [],
                "poser": "cobblemon:standard",
                "model": "cobblemon:trainer.geo",
                "texture": f"pokemonleague:textures/npcs/{npc_id}.png",
                "layers": []
            }
        ]
    }
    return variation

def create_npc_definition(npc_id, name, skill):
    """Create NPC definition JSON file"""
    npc = {
        "resourceIdentifier": f"pokemonleague:{npc_id}",
        "hitbox": "player",
        "names": [name],
        "battleConfiguration": {
            "canChallenge": True
        },
        "skill": skill
    }
    return npc

def main():
    # Create directories
    os.makedirs(NPCS_PATH, exist_ok=True)

    for npc in NPCS:
        npc_id = npc["id"]
        name = npc["name"]
        skill = npc["skill"]

        # Create variation directory and file
        var_dir = os.path.join(VARIATIONS_PATH, npc_id)
        os.makedirs(var_dir, exist_ok=True)

        var_file = os.path.join(var_dir, f"0_{npc_id}.json")
        with open(var_file, 'w', encoding='utf-8') as f:
            json.dump(create_variation(npc_id), f, indent=2)
        print(f"Created: {var_file}")

        # Create NPC definition file
        npc_file = os.path.join(NPCS_PATH, f"{npc_id}.json")
        with open(npc_file, 'w', encoding='utf-8') as f:
            json.dump(create_npc_definition(npc_id, name, skill), f, indent=2)
        print(f"Created: {npc_file}")

    print(f"\nGenerated {len(NPCS)} NPC configurations!")
    print("\nN'oublie pas d'ajouter les fichiers PNG dans:")
    print("  assets/pokemonleague/textures/npcs/")

if __name__ == "__main__":
    main()
